const express = require("express");
const router = express.Router();
const fs = require("fs");
const axios = require("axios");
const ffmpegPath = require("@ffmpeg-installer/ffmpeg").path;
const ffprobePath = require("@ffprobe-installer/ffprobe").path;
const ffmpeg = require("fluent-ffmpeg");
const videoshow = require("videoshow");
const AWS = require("aws-sdk");
const path = require("path");
const { uuid } = require("uuidv4");
const sharp = require("sharp");

const { v4: uuidv4 } = require("uuid"); //
const multer = require("multer"); // add multer
const { upload } = require("../multer/input_audio"); // add audio file
const { fileupload } = require("../multer/fileUpload");
const { Videofileupload } = require("../multer/Videofileupload");
const { checkOutputVideoPathExist } = require("./helper");

const outputDirectory = path.join(__dirname, "..", "output_video");

ffmpeg.setFfmpegPath(ffmpegPath);
ffmpeg.setFfprobePath(ffprobePath);

// Configure AWS SDK
const s3 = new AWS.S3({
  // Set your AWS credentials and region
  accessKeyId: process.env.AWS_ACCESS_KEY_ID,
  secretAccessKey: process.env.AWS_SECRET_ACCESS_KEY,
});

/**
 * @function formatStringToSec
 * @description Formats String-{hh:mm:ss} to seconds format
 * @param {String} 'hh:mm:ss'
 * @returns {Number} seconds
 */
function formatStringToSec(timeString = "00:00:00") {
  //param should be in format HH:MM:SS
  let seconds = 0;
  timeString
    .trim()
    .split(":")
    .reverse()
    .forEach((time, index) => {
      if (index === 0) {
        seconds += +time;
      } else {
        seconds += +time * (index * 60);
      }
    });
  console.log(
    "HH:MM:SS -> " + timeString + " => converted to seconds : " + seconds
  );
  return seconds;
}

/**
 * @function downloadImage
 * @description Downloads an image from a given URL to the specified path
 * @param {String} imageUrl - URL of the image to be downloaded
 * @param {String} imagePath - Path where the image will be saved
 * @returns {Promise} - Returns a Promise that resolves when the image is downloaded successfully
 */
async function downloadImage(imageUrl, imagePath) {
  const response = await axios({
    method: "GET",
    url: imageUrl,
    responseType: "stream",
  });

  return new Promise((resolve, reject) => {
    const stream = response.data.pipe(fs.createWriteStream(imagePath));
    stream.on("finish", () => resolve(imagePath)); // Resolve with imagePath
    stream.on("error", reject);
  });
}

/**
 * @function resizeVideo
 * @description Resizes a video to the specified dimensions
 * @param {String} videoPath - Path of the video to resize
 * @param {Number} width - Desired width of the resized video
 * @param {Number} height - Desired height of the resized video
 * @returns {Promise} - Returns a Promise that resolves when the video is resized successfully
 */
async function resizeVideo(videoPath, width, height, ind) {
  console.log("TEst", videoPath);
  return new Promise((resolve, reject) => {
    const outputVideoPath = `./output_video/${ind}_resized.mp4`;
    ffmpeg(videoPath)
      .outputOptions(`-vf scale=${width}:${height}`)
      .on("end", () => resolve(outputVideoPath))
      .on("error", (err) => reject(err))
      .save(outputVideoPath);
  });
}

/**
 * @function resizeImage
 * @description Resizes an image to the specified dimensions
 * @param {String} imagePath - Path of the image to resize
 * @param {Number} width - Desired width of the resized image
 * @param {Number} height - Desired height of the resized image
 * @returns {Promise} - Returns a Promise that resolves when the image is resized successfully
 */

async function resizeImage(imagePath, width, height) {
  return sharp(imagePath)
    .resize({ width, height })
    .toBuffer()
    .then((data) => {
      fs.writeFileSync(imagePath, data); // Overwrite the original image with the resized one
    });
}

/**
 * @function uploadToS3
 * @description Uploads a file to an S3 bucket
 * @param {String} filePath - Path of the file to upload
 * @param {String} bucketName - Name of the S3 bucket
 * @param {String} key - Key (path) under which to store the file in the bucket
 * @returns {Promise} - Returns a Promise that resolves when the file is uploaded successfully
 */
async function uploadToS3(filePath, name) {
  try {
    const fileContent = fs.createReadStream(filePath);
    const params = {
      Bucket: process.env.AWS_BUCKET_NAME,
      Key: name,
      Body: fileContent,
      ACL: "public-read",
    };

    return new Promise((resolve, reject) => {
      s3.upload(params, (err, data) => {
        if (err) {
          console.error("Error uploading video:", err);
          reject(err);
          return;
        }
        console.log("Video uploaded successfully:", data);
        resolve(data);
      });
    });
  } catch (err) {
    console.error("Error:", err);
    throw err;
  }
}

// Define Express route
router.post("/images/create-video", async (req, res) => {
  // Extract data from request body or query parameters
  const { imageUrls } = req.body;

  // Define constants
  const imagePaths = [];
  const imageWidth = 640; // Set desired width for resized images
  const imageHeight = 480; // Set desired height for resized images

  try {
    // Download and resize images
    for (let i = 0; i < imageUrls.length; i++) {
      const imagePath = `./input_video/image_${uuid()}.jpg`; // Assuming all images will be JPEG format
      const downloadedImagePath = await downloadImage(imageUrls[i], imagePath);
      // await resizeImage(downloadedImagePath, imageWidth, imageHeight);
      imagePaths.push(downloadedImagePath);
    }

    // Define video options
    const videoOptions = {
      fps: 25,
      transition: true,
      transitionDuration: 1,
      videoBitrate: 1024,
      videoCodec: "libx264",
      size: `${imageWidth}x${imageHeight}`,
      audioBitrate: "128k",
      audioChannels: 2,
      format: "mp4",
      pixelFormat: "yuv420p",
      loop: 3,
    };

    // Create video
    const name = `${Date.now()}.mp4`;
    const outputVideoPath = path.join(outputDirectory, `${name}`);
    await videoshow(imagePaths, videoOptions).save(outputVideoPath);
    console.log(imagePaths);

    await new Promise((resolve) =>
      setTimeout(resolve, imageUrls.length * 7000)
    ); // Adjust the delay time as needed


    const exists = await checkOutputVideoPathExist(outputVideoPath);
    if (exists) {
      // Upload video to S3 bucket
      const bucketResponse = await uploadToS3(outputVideoPath, name);

      res.status(200).json({
        message: "Video created and uploaded successfully",
        videoURL: bucketResponse.Location,
      });
    } else {
      console.error("Error:", err);
      res.status(500).json({ error: "An error occurred" });
    }
  } catch (err) {
    console.error("Error:", err);
    res.status(500).json({ error: "An error occurred" });
  }
});

function generateGrid(videosLength) {
  let gridFilter = "[";
  for (let i = 0; i < videosLength; i++) {
    gridFilter += `[a${i}]`;
  }
  gridFilter += `hstack=${videosLength}[outv]`;
  return gridFilter;
}

function generateComplexFilter(videosLength) {
  const filter = [];
  for (let i = 0; i < videosLength; i++) {
    filter.push(`[${i}:v] scale=qvga [a${i}];`);
  }
  filter.push(generateGrid(videosLength));
  return `"${filter.join("")}"`;
}

/**
 * @function mergeVideos
 * @description Merges multiple videos using ffmpeg
 * @param {Array} videoPaths - Array of paths of the input videos
 * @param {String} outputVideoPath - Path where the merged video will be saved
 * @returns {Promise} - Returns a Promise that resolves when the videos are merged successfully
 */

async function mergeVideos(videoPaths, outputVideoPath) {
  return new Promise((resolve, reject) => {
    const command = ffmpeg();
    let filterString = ""; // Initialize an empty filter string

    // Build the filter string dynamically based on the number of video paths
    for (let i = 0; i < videoPaths.length; i++) {
      filterString += `[${i}:v]`; // Append input video index to filter string
    }

    filterString += `concat=n=${videoPaths.length}:v=1:a=0[v]`; // Concatenate filter

    console.log(filterString);
    videoPaths.forEach((path, index) => {
      command.input(path);
    });
    // Configure ffmpeg command
    command
      .inputOptions(["-hide_banner"]) // Hide ffmpeg banner
      .complexFilter([filterString]) // Apply dynamic filter
      .outputOptions(["-map [v]"]) // Map video stream
      .on("end", () => resolve(outputVideoPath)) // Resolve promise on completion
      .on("error", (err) => reject(err)) // Reject promise on error
      .save(outputVideoPath); // Save output video
  });
}

router.post("/videos/create-video", async (req, res) => {
  // Extract data from request body or query parameters

  const { imageUrls } = req.body;
 console.log(imageUrls);
  const videoPaths = [];
  const videoWidth = 640; // Set desired width for resized images
  const videoHeight = 480; // Set desired height for resized images

  try {
    // Download and resize images
    for (let i = 0; i < videoUrls.length; i++) {
      const videoPath = `./input_video/video_${i}.${videoUrls[i]
        .split(".")
        .pop()}`; // Assuming all videos will be in MP4 format
      await downloadImage(videoUrls[i], videoPath);
      videoPaths.push(videoPath);
    }

    // // Resize videos
    // const resizedVideoPaths = [];
    // for (let i = 0; i < videoPaths.length; i++) {
    //   const resizedVideoPath = await resizeVideo(
    //     videoPaths[i],
    //     videoWidth,
    //     videoHeight,
    //     i
    //   ); // Set desired width and height
    //   resizedVideoPaths.push(resizedVideoPath);
    // }

    // Create video
    const name = `${Date.now()}.mp4`;
    const outputVideoPath = path.join(outputDirectory, `${name}`);
    mergeVideos(videoUrls, outputVideoPath)
      .then((outputPath) => {
        console.log("Videos merged successfully. Output:", outputPath);
      })
      .catch((error) => {
        console.error("Error merging videos:", error);
      });

    await new Promise((resolve) => setTimeout(resolve, 2000)); // Adjust the delay time as needed

    // Upload video to S3 bucket
    // await uploadToS3(outputVideoPath, name)
    //   .then((e) => console.log("Test Successfuly", e))
    //   .catch((e) => console.log(e));

    // Send success response
    res
      .status(200)
      .json({ message: "Video created and uploaded successfully" });
  } catch (err) {
    console.error("Error:", err);
    res.status(500).json({ error: "An error occurred" });
  }
});

// Route function for uploading videos
router.post(
  "/videos/upload-videos",
  Videofileupload.array("videos"),
  async (req, res) => {
    try {
      if (!req.files || req.files.length === 0) {
        return res.status(400).json({ message: "No files uploaded" });
      }

      const uploadedFiles = [];

      // Iterate through each uploaded video
      for (const file of req.files) {
        const videoFilePath = file.path;
        const key = `videos/${Date.now()}_${file.originalname}`;

        // Upload the video file to S3
        const videoUrl = await uploadToS3(videoFilePath, key);

        uploadedFiles.push({ filename: file.originalname, url: videoUrl });
      }

      // Send the success response with the uploaded videos' filenames and S3 URLs
      res
        .status(200)
        .json({
          message: "Videos uploaded successfully",
          files: uploadedFiles,
        });
    } catch (error) {
      res.status(500).json({ error: error.message });
    }
  }
);

// Route function for uploading audio files By Maha
 router.post("/videos/upload-audio",upload.single("audioFile"),async (req, res) => {
    try {
      if (!req.file) {
        return res.status(400).json({ message: "No file uploaded" });
      }

      // Upload the audio file to S3
      const audioFilePath = req.file.path;
      const videoFilePath = req.body.videoFile;
      const outputFileName = `./output_VideoWithAudio/${uuid()}.mp4`; // Generate unique file name

      // Call the mergeVideoWithAudio function
      await mergeVideoWithAudio(videoFilePath, audioFilePath, outputFileName);

      const key = `output_VideoWithAudio/${Date.now()}_${outputFileName}`;
      const bucketResponse = await uploadToS3(outputFileName ,key);

      // Send the success response with the audio filename and S3 URL
      res.status(200).json({
          message: "Audio is merge uploaded successfully",
          mergeVideoWithAudio: bucketResponse.Location,
        });
    } catch (error) {
      res.status(500).json({ error: error.message });
    }
  }
);

// Route function for uploading Images from Computer files By Maha

router.post("/videos/upload-images",fileupload.array("files"),async (req, res) => {

    try {
      const imagePaths = req.files.map((file) => file.path);
      const imageWidth = 580; // Set desired width for resized images
      const imageHeight = 580; // Set desired height for resized images
      const resizedImagePath = [];

      imagePaths.forEach(async (element) => {
        await resizeImage(element, imageWidth, imageHeight);
      });

      console.log(imagePaths, "Get images");
      // Define video options
      const videoOptions = {
        fps: 25,
        transition: true,
        transitionDuration: 1,
        videoBitrate: 1024,
        videoCodec: "libx264",
        size: `${imageWidth}x${imageHeight}`,
        audioBitrate: "128k",
        audioChannels: 2,
        format: "mp4",
        pixelFormat: "yuv420p",
        loop: 3,
      };

      // Create video
      const name = `${Date.now()}.mp4`;
      const outputVideoPath = path.join(outputDirectory, `${name}`);

      await videoshow(imagePaths, videoOptions).save(outputVideoPath);

      if (imagePaths && imagePaths.length > 0) {
        await new Promise((resolve) =>
          setTimeout(resolve, imagePaths.length * 7000)
        ); // Adjust the delay time as needed

        console.error("set time ");
      } else {
        console.error("imagePaths is undefined or empty");
      }

      const exists = await checkOutputVideoPathExist(outputVideoPath);
      if (exists) {
        // Proceed with uploading or any other operation
        const bucketResponse = await uploadToS3(outputVideoPath, name);

        if (bucketResponse) {
          res.status(200).json({
            message: "Video created and uploaded successfully",
            videoURL: bucketResponse.Location,
            outputVideoPath: outputVideoPath,
          });
        } else {
          res
            .status(500)
            .json({ error: "S3 upload failed or response is invalid" });
        }
      } else {
        console.error("Output video path does not exist.");
      }

      // Upload video to S3 bucket
      // const bucketResponse = await uploadToS3(outputVideoPath, name);
      // const bucketResponse = await uploadToS3(outputVideoPath, name)
      // .then((response) => {
      //   if (response && response.Location) {
      //     return response;
      //   } else {
      //     throw new Error("S3 upload failed or response is invalid");
      //   }
      // })
      // .catch((error) => {
      //   console.error("Error uploading to S3:", error);
      //   return null; // or any default response you want to return
      // });
    } catch (error) {
      res.status(500).json({ error: error.message });
    }
  }
);

// Function to merge video with audio by Maha
router.post("/video/merge-video-with-audio", async (req, res) => {
  try {
    // Define paths for input video, audio, and output video
    const inputVideoPath = "./output_video/1712095996851.mp4"; // Adjust this path
    const audioPath = "./input_audio/mp3.mp3"; // Adjust this path

    // Call the mergeVideoWithAudio function
    const outputFileName = `./output_VideoWithAudio/${uuid()}.mp4`; // Generate unique file name
    await mergeVideoWithAudio(inputVideoPath, audioPath, outputFileName);
    // Send success response
    res.status(200).json({ message: "Audio merged with video successfully" });
  } catch (error) {
    res.status(500).json({
      error: error.message,
    });
  }
});

// Function to merge video with audio by Maha
async function mergeVideoWithAudio(inputVideoPath, audioPath, outputFileName) {
  return new Promise((resolve, reject) => {
    ffmpeg()
      .addInput(inputVideoPath)
      .addInput(audioPath)
      .output(outputFileName)
      .outputOptions([
        "-map 0:v",
        "-map 1:a",
        "-c:v copy",
        "-c:a copy",
        "-shortest",
      ]) // Add -c:a copy to copy audio codec

      .on("end", () => {
        console.log("Audio merged with video successfully");
        resolve();
      })
      .on("error", (err) => {
        console.error("Error merging audio with video:", err);
        reject(err);
      })
      .run();
  });
}

module.exports = router;